program QUADRATICMIN;
{--------------------------------------------------------------------}
{  Alg8'3.pas   Pascal program for implementing Algorithm 8.3        }
{                                                                    }
{  NUMERICAL METHODS: Pascal Programs, (c) John H. Mathews 1995      }
{  To accompany the text:                                            }
{  NUMERICAL METHODS for Math., Science & Engineering, 2nd Ed, 1992  }
{  Prentice Hall, Englewood Cliffs, New Jersey, 07632, U.S.A.        }
{  Prentice Hall, Inc.; USA, Canada, Mexico ISBN 0-13-624990-6       }
{  Prentice Hall, International Editions:   ISBN 0-13-625047-5       }
{  This free software is compliments of the author.                  }
{  E-mail address:       in%"mathews@fullerton.edu"                  }
{                                                                    }
{  Algorithm 8.3 (Local Minimum Search: Quadratic Interpolation).    }
{  Section   8.1, Minimization of a Function, Page 416               }
{--------------------------------------------------------------------}

  uses
    crt;

  const
    Jmax = 50;
    Kmax = 50;

  type
    LETTERS = string[100];
    AVECTOR = array[0..200] of real;
    STATUS = (Computing, Done, Working);

  var
    Cond, Count, FunType, Inum, Sub: integer;
    Delta, Epsilon, Error, H, P0, Pmin, Rnum, Y0, Ymin: real;
    VA: AVECTOR;
    Ans: CHAR;
    Mess: LETTERS;
    Stat, State: STATUS;

  function F (X: real): real;
  begin
    case FunType of
      1: 
        F := X * X - SIN(X);
      2: 
        F := 3 * X * X - 2 * X + 5;
      3: 
        F := 2 * X * X * X - 3 * X * X - 12 * X + 1;
      4: 
        F := 4 * X * X * X - 8 * X * X - 11 * X + 5;
      5: 
        F := X + 3 / (X * X);
      6: 
        F := (X + 2.5) / (4 - X * X);
      7: 
        F := EXP(X) / (X * X);
      8: 
        F := -SIN(X) - SIN(3 * X) / 3;
      9: 
        F := -2 * SIN(X) + SIN(2 * X) - 2 * SIN(3 * X) / 3;
    end;
  end;

  function F1 (X: real): real;
  begin
    case FunType of
      1: 
        F1 := 2 * X - COS(X);
      2: 
        F1 := 6 * X - 2;
      3: 
        F1 := 6 * X * X - 6 * X - 12;
      4:
        F1 := 12 * X * X - 16 * X - 11;
      5: 
        F1 := 1 - 6 / (X * X * X);
      6: 
        F1 := (X * X + 5 * X + 4) / ((4 - X * X) * (4 - X * X));
      7: 
        F1 := EXP(X) / (X * X) - 2 * EXP(X) / (X * X * X);
      8: 
        F1 := -COS(X) - COS(3 * X);
      9: 
        F1 := -2 * COS(X) + 2 * COS(2 * X) - 2 * COS(3 * X);
    end;
  end;

  procedure PRINTFUN (FunType: integer);
  begin
    case FunType of
      1: 
        WRITELN('X*X - SIN(X)');
      2: 
        WRITELN('3*X*X - 2*X + 5');
      3: 
        WRITELN('2*X*X*X - 3*X*X - 12*X + 1');
      4: 
        WRITELN('4*X*X*X - 8*X*X - 11*X + 5');
      5: 
        WRITELN('X + 3/(X*X)');
      6: 
        WRITELN('(X + 2.5)/(4 - X*X)');
      7: 
        WRITELN('EXP(X)/(X*X)');
      8: 
        WRITELN('- SIN(X) - SIN(3*X)/3');
      9:
        WRITELN('- 2*SIN(X) + SIN(2*X) - 2*SIN(3*X)/3');
    end;
  end;

  procedure QMIN ( {FUNCTION F(X: real): real;}
                  P0, Delta, Epsilon: real; Jmax, Kmax: integer; var Pmin, H, Ymin, Error: real; var Cond, Count: integer);
    label
      999;
    const
      Big = 1E9;
    var
      J, K: integer;
      D, E0, E1, E2, H0, H1, H2, Hmin, P1, P2, Y0, Y1, Y2: real;
  begin
    K := 0;
    Error := 1;
    Cond := 0;
    Count := 0;
    H := 1;
    VA[0] := P0;   {Store first approximation}
    if ABS(P0) > 1E4 then
      H := ABS(P0) / 1E4;
    while (K < Kmax) and (Error > Epsilon) and (Cond <> 5) do
      begin
        if F1(P0) > 0 then
          H := -ABS(H);
        P1 := P0 + H;
        P2 := P0 + 2 * H;
        Pmin := P0;
        Y0 := F(P0);
        Y1 := F(P1);
        Y2 := F(P2);
        Ymin := Y0;
        Cond := 0;
        J := 0;
        while (J < Jmax) and (ABS(H) > Delta) and (Cond = 0) do
          begin
            if Y0 <= Y1 then
              begin
                P2 := P1;
                Y2 := Y1;
                H := H / 2;
                P1 := P0 + H;
                Y1 := F(P1);
              end
            else
              begin
                if Y2 < Y1 then
                  begin
                    P1 := P2;
                    Y1 := Y2;
                    H := 2 * H;
                    P2 := P0 + 2 * H;
                    Y2 := F(P2);
                  end
                else
                  Cond := -1;
              end;
            J := J + 1;
            if (ABS(H) > Big) or (ABS(P0) > Big) then
              Cond := 5;
          end;
        if Cond = 5 then
          begin
            Pmin := P1;
            Ymin := F(P1);
            goto 999;
          end;
        D := 4 * Y1 - 2 * Y0 - 2 * Y2;
        if D < 0 then
          Hmin := H * (4 * Y1 - 3 * Y0 - Y2) / D
        else
          begin
            Hmin := H / 3;
            Cond := 4;
          end;
        Pmin := P0 + Hmin;
        Ymin := F(Pmin);
        H := ABS(H);
        H0 := ABS(Hmin);
        H1 := ABS(Hmin - H);
        H2 := ABS(Hmin - 2 * H);
        if H0 < H then
          H := H0;
        if H1 < H then
          H := H1;
        if H2 < H then
          H := H2;
        if H = 0 then
          H := Hmin;
        if H < Delta then
          Cond := 1;
        if (ABS(H) > Big) or (ABS(Pmin) > Big) then
          Cond := 5;
        E0 := ABS(Y0 - Ymin);
        E1 := ABS(Y1 - Ymin);
        E2 := ABS(Y2 - Ymin);
        if (E0 <> 0) and (E0 < Error) then
          Error := E0;
        if (E1 <> 0) and (E1 < Error) then
          Error := E1;
        if (E2 <> 0) and (E2 < Error) then
          Error := E2;
        if (E0 = 0) and (E1 = 0) and (E2 = 0) then
          Error := 0;
        if Error < Epsilon then
          Cond := 2;
        P0 := Pmin;
        K := K + 1;
        VA[K] := P0;   {Store approximations}
999:
      end;
    Count := Count + K;
    if (Cond = 2) and (H < Delta) then
      Cond := 3;
  end;

  procedure GETFUN (var FunType: integer);
    var
      K: integer;
  begin
    CLRSCR;
    WRITELN;
    WRITELN('     You have a choice of functions:');
    WRITELN;
    for K := 1 to 9 do
      begin
        WRITE('     <', K : 2, ' >  F(X) = ');
        PRINTFUN(K);
        WRITELN;
      end;
    Mess := '     Select < 1 - 9 > ?  ';
    FunType := 1;
    WRITE(Mess);
    READLN(FunType);
    if FunType < 1 then
      FunType := 1;
    if FunType > 9 then
      FunType := 9;
  end;

  procedure INPUT (var Delta, Epsilon, P0: real);
  begin
    CLRSCR;
    WRITELN;
    WRITELN('          You chose the function:');
    WRITELN;
    WRITE('          ');
    PRINTFUN(FunType);
    WRITELN;
    WRITELN('     The quadratic search method is used to find a local minimum.');
    WRITELN;
    WRITELN('     One initial approximation  p  is needed.');
    WRITELN('                                 0');
    WRITELN;
    WRITELN;
    Mess := '     ENTER the starting value   p0 = ';
    P0 := 0;
    WRITE(Mess);
    READLN(P0);
    Delta := 1E-6;
    Mess := '     ENTER  the  value for   Delta = ';
    WRITE(Mess);
    READLN(Delta);
    Delta := ABS(Delta);
    if Delta > 0.1 then
      Delta := 0.1;
    if Delta < 1E-6 then
      Delta := 1E-6;
    Mess := '     ENTER the value for   Epsilon = ';
    Epsilon := 1E-8;
    WRITE(Mess);
    READLN(Epsilon);
    Epsilon := ABS(Epsilon);
    if Epsilon > 0.1 then
      Epsilon := 0.1;
    if Epsilon < 1E-8 then
      Epsilon := 1E-8;
    WRITELN;
  end;

  procedure RESULT (P0, Pmin, H, Ymin, Error, Epsilon: real; Cond, Count: integer);
  begin
    CLRSCR;
    WRITELN;
    WRITELN('The quadratic search method was used to find an');
    WRITELN;
    WRITELN('approximation to a local minimum of the function: ');
    WRITELN;
    PRINTFUN(FunType);
    WRITELN;
    WRITELN('The starting approximation was  p  =', P0 : 15 : 7);
    WRITELN('                                 0');
    WRITELN;
    WRITELN('After ', Count : 2, ' iterations an approximation for a minimum is:');
    WRITELN;
    WRITELN('     P  =', Pmin : 15 : 7);
    WRITELN;
    WRITELN('    DP  =', ABS(H) : 15 : 7, '  is the estimated accuracy for P.');
    WRITELN;
    WRITELN('       F(', Pmin : 15 : 7, '  )  =', Ymin : 15 : 7);
    WRITELN;
    case Cond of
      0:
        begin
          WRITELN('Convergence is doubtful because the');
          WRITELN;
          WRITE('maximum number of iterations was exceeded.');
        end;
      1: 
        begin
          WRITELN('Convergence has been achieved because ');
          WRITELN;
          WRITE('consecutive abscissas are closer than ', Delta : 15 : 7);
        end;
      2: 
        begin
          WRITELN('Convergence has been achieved because ');
          WRITELN;
          WRITE('consecutive ordinates are closer than ', Epsilon : 15 : 7);
        end;
      3: 
        begin
          WRITELN('Convergence has been achieved because');
          WRITELN;
          WRITELN('the consecutive abscissas are closer than ', H : 15 : 7);
          WRITELN;
          WRITE('and consecutive ordinates are closer than ', Error : 15 : 7);
        end;
      4: 
        begin
          WRITELN('Convergence is doubtful because division by zero was encountered.');
          if Error < Epsilon / 100 then
            begin
              WRITELN;
              WRITE('However, the consecutive ordinates are close.');
            end;
        end;
      5: 
        begin
          WRITELN('Convergence is doubtful, H is too large, H  =', H : 15 : 7);
          WRITELN;
          WRITELN('Perhaps a different starting value should be used.');
          WRITELN;
          WRITE('It is possible that there is no local minimum.');
        end;
    end;
  end;

  procedure MESSAGE;
  begin
    CLRSCR;
    WRITELN;
    WRITELN('                 SEARCH FOR A LOCAL MINIMUM');
    WRITELN;
    WRITELN;
    WRITELN('    Search for a local minimum using quadratic interpolation. It is assumed');
    WRITELN;
    WRITELN;
    WRITELN('that  f(x)  and  f`(x)  are defined and that  f(x)  has a local minimum on');
    WRITELN;
    WRITELN;
    WRITELN('either the interval  [A,p ]  or  [p ,B].  The sign of  f`(p )  is used to');
    WRITELN('                         0         0                       0  ');
    WRITELN;
    WRITELN('determine which interval to search, then two additional points p  and  p ');
    WRITELN('                                                                1       2');
    WRITELN;
    WRITELN('are selected and a parabola is fit through the three points.  The minimum');
    WRITELN;
    WRITELN;
    WRITELN('point on the parabola is used to continue the process.');
    WRITELN;
    WRITELN;
    WRITE('                    Press the <ENTER> key.  ');
    READLN(Ans);
    WRITELN;
  end;

  procedure PRINTAPPROXS;
    var
      K: integer;
  begin
    CLRSCR;
    WRITELN;
    WRITELN('  k             p             ');
    WRITELN('                 k            ');
    WRITELN('------------------------------');
    for K := 0 to Count do
      begin
        WRITELN;
        WRITELN(' ', K : 2, '     ', VA[K] : 15 : 7);
      end;
    WRITELN;
    WRITE('Press  the  <ENTER> key.  ');
    READLN(Ans);
    WRITELN;
    WRITELN;
  end;

begin                                            {Begin Main Program}
  Stat := Working;
  MESSAGE;
  while (Stat = Working) do
    begin
      GETFUN(FunType);
      State := Computing;
      while (State = Computing) do
        begin
          INPUT(Delta, Epsilon, P0);
          QMIN(P0, Delta, Epsilon, Jmax, Kmax, Pmin, H, Ymin, Error, Cond, Count);
          RESULT(P0, Pmin, H, Ymin, Error, Epsilon, Cond, Count);
          WRITELN;
          WRITELN;
          WRITE('Want  to see  all of  the approximations ?  <Y/N>  ');
          READLN(Ans);
          WRITELN;
          if (Ans = 'Y') or (Ans = 'y') then
            PRINTAPPROXS
          else
            WRITELN;
          WRITE('Want to try  a different  starting value ?  <Y/N>  ');
          READLN(Ans);
          WRITELN;
          if (Ans <> 'Y') and (Ans <> 'y') then
            State := Done;
          if (Ans = 'Y') or (Ans = 'y') then
            CLRSCR;
        end;
      WRITELN;
      WRITE('Want  to try  a different function  F(X) ?  <Y/N>  ');
      READLN(Ans);
      WRITELN;
      if (Ans <> 'Y') and (Ans <> 'y') then
        Stat := Done
    end;
end.                                               {End Main Program}

